<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Controllers\Traits\CsvImportTrait;
use App\Http\Requests\MassDestroySchReportRequest;
use App\Http\Requests\StoreSchReportRequest;
use App\Http\Requests\UpdateSchReportRequest;
use App\Models\SchClass;
use App\Models\SchReport;
use App\Models\Schstudent;
use App\Models\SchSubject;
use App\Models\SchTeacher;
use Gate;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class SchReportController extends Controller
{
    use CsvImportTrait;

    public function index()
    {
        abort_if(Gate::denies('sch_report_access'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        $schReports = SchReport::with(['schclass', 'schstudent', 'schteacher', 'schsubject'])->get();

        $sch_classes = SchClass::get();

        $schstudents = Schstudent::get();

        $sch_teachers = SchTeacher::get();

        $sch_subjects = SchSubject::get();

        return view('admin.schReports.index', compact('schReports', 'sch_classes', 'sch_subjects', 'sch_teachers', 'schstudents'));
    }

    public function create(Request $request)
    {
        abort_if(Gate::denies('sch_report_create'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        $classId = $request->input('class_id'); // Отримуємо class_id з URL

        $schclasses = SchClass::pluck('name', 'id')->prepend(trans('global.pleaseSelect'), '');

        $schstudents = Schstudent::pluck('name_ua', 'id')->prepend(trans('global.pleaseSelect'), '');

        $schteachers = SchTeacher::pluck('name_ua', 'id')->prepend(trans('global.pleaseSelect'), '');

        $schsubjects = SchSubject::pluck('name', 'id')->prepend(trans('global.pleaseSelect'), '');

        return view('admin.schReports.create', compact('schclasses', 'schstudents', 'schsubjects', 'schteachers', 'classId'));
    }

public function store(StoreSchReportRequest $request)
{
    $data = $request->all();
    $schTeacher = \App\Models\SchTeacher::where('user_id', auth()->id())->first();
    $teacherId = $schTeacher ? $schTeacher->id : null;

    // Преобразуем названия полей
    $data['schclass_id'] = $data['class_id'];
    $data['schstudent_id'] = $data['student_id'];
    $data['schteacher_id'] = $teacherId;
    $data['schsubject_id'] = $data['subject_id'];

    // Создаем репорт один раз
    SchReport::create($data);

    return redirect()->back()->with('success', 'Репорт успішно додано!');
}

    public function edit(SchReport $schReport)
    {
        abort_if(Gate::denies('sch_report_edit'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        $schclasses = SchClass::pluck('name', 'id')->prepend(trans('global.pleaseSelect'), '');

        $schstudents = Schstudent::pluck('name_ua', 'id')->prepend(trans('global.pleaseSelect'), '');

        $schteachers = SchTeacher::pluck('name_ua', 'id')->prepend(trans('global.pleaseSelect'), '');

        $schsubjects = SchSubject::pluck('name', 'id')->prepend(trans('global.pleaseSelect'), '');

        $schReport->load('schclass', 'schstudent', 'schteacher', 'schsubject');

        return view('admin.schReports.edit', compact('schReport', 'schclasses', 'schstudents', 'schsubjects', 'schteachers'));
    }

    public function update(UpdateSchReportRequest $request, SchReport $schReport)
    {
        $data = $request->only(['report', 'report_en', 'schsubject_id']);

        // Проверяем права доступа
        if ($schReport->schteacher_id != auth()->user()->schTeacher->id) {
            abort(403, 'Unauthorized action.');
        }

        $schReport->update($data);

        return response()->json([
            'success' => true,
            'message' => 'Репорт успішно оновлено'
        ]);
    }

    public function show(SchReport $schReport)
    {
        abort_if(Gate::denies('sch_report_show'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        $schReport->load('schclass', 'schstudent', 'schteacher', 'schsubject');

        return view('admin.schReports.show', compact('schReport'));
    }

    public function destroy(SchReport $schReport)
    {
        abort_if(Gate::denies('sch_report_delete'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        $schReport->delete();

        return back();
    }

    public function massDestroy(MassDestroySchReportRequest $request)
    {
        $schReports = SchReport::find(request('ids'));

        foreach ($schReports as $schReport) {
            $schReport->delete();
        }

        return response(null, Response::HTTP_NO_CONTENT);
    }


    public function getClassData(Request $request)
    {
        $classId = $request->input('class_id');
        $user = auth()->user();
        $schTeacher = \App\Models\SchTeacher::where('user_id', $user->id)->first();

        $students = \App\Models\Schstudent::where('schclass_id', $classId)
            ->orderBy('name_ua')
            ->get(['id', 'name_ua as name']);

        foreach ($students as $student) {
            $student->reports_count = \App\Models\SchReport::where('schstudent_id', $student->id)
                ->where('schclass_id', $classId)
                ->count();
        }

        // Отримуємо предмети, які належать до класу
        $subjects = \App\Models\SchSubject::where('sch_class_id', $classId)
            ->pluck('name', 'id');

        if ($students->isEmpty()) {
            return response()->json(['error' => 'Учні не знайдені'], Response::HTTP_NOT_FOUND);
        }

        return response()->json([
            'students' => $students,
            'subjects' => $subjects,
        ]);
    }

    public function classList()
    {
        $user = auth()->user();
        $schTeacher = \App\Models\SchTeacher::where('user_id', $user->id)->first();

        // Добавляем отладочную информацию
        \Log::info('User ID: ' . $user->id);
        \Log::info('Teacher: ' . ($schTeacher ? 'Found (ID: ' . $schTeacher->id . ')' : 'Not found'));

        if (!$schTeacher) {
            $schclasses = \App\Models\SchClass::orderBy('name')->get();
            \Log::info('Admin view - showing all classes');
        } else {
            // Получаем ID предметов учителя
            $teacherSubjectIds = $schTeacher->schsubjects()
                ->pluck('sch_subjects.id')
                ->toArray();

            \Log::info('Teacher subjects: ' . implode(', ', $teacherSubjectIds));

            // Получаем классы где есть эти предметы
            $classIds = \App\Models\SchSubject::whereIn('id', $teacherSubjectIds)
                ->pluck('sch_class_id')
                ->unique()
                ->toArray();

            \Log::info('Classes found: ' . implode(', ', $classIds));

            $schclasses = \App\Models\SchClass::whereIn('id', $classIds)
                ->orderBy('name')
                ->get();
        }

        return view('admin.schReports.class_list', compact('schclasses'));
    }

    // SchReportController.php
    public function studentReports(Request $request)
    {
        $studentId = $request->input('student_id');
        $classId = $request->input('class_id');

        // Добавим логирование для отладки
        \Log::info('Student Reports Request', [
            'student_id' => $studentId,
            'class_id' => $classId
        ]);

        $schTeacher = \App\Models\SchTeacher::where('user_id', auth()->id())->first();
        $teacherId = $schTeacher ? $schTeacher->id : null;
        $teacherUserId = auth()->id(); // Добавляем ID пользователя
        $schTeacherName = $schTeacher ? $schTeacher->name_ua : '';

        // Отримуємо репорти
        $reports = \App\Models\SchReport::where('schstudent_id', $studentId)
            ->where('schclass_id', $classId)
            ->with('schteacher')
            ->get();

        \Log::info('Reports found:', ['count' => $reports->count()]);

        // Отримуємо ПІБ учня
        $student = \App\Models\Schstudent::find($studentId);
        $studentName = $student ? $student->name_ua : '';

        // Предмети цього класу
        $sch_subjects = $schTeacher
            ? \App\Models\SchSubject::where('sch_class_id', $classId)
            ->whereHas('schsubjectSchTeachers', function ($q) use ($schTeacher) {
                $q->where('sch_teacher_id', $schTeacher->id);
            })
            ->pluck('name', 'id')
            : \App\Models\SchSubject::where('sch_class_id', $classId)->pluck('name', 'id');

        return view('admin.schReports.student_reports', compact(
            'reports',
            'studentId',
            'classId',
            'sch_subjects',
            'teacherId',
            'teacherUserId',
            'schTeacherName',
            'studentName'
        ))->render(); // Добавляем render() для AJAX-запроса
    }
}
